	function gencslp(datfile, lpspec)
	%gencslp(datfile, lpspec)
	%
	%Generates the data file for an ILP optimization problem in the GNU
	%Mathprog modeling language
	%
	%Arguments:
	%  datfile - the name of the file to generate
	%  lpspec: structure with fields:
	%    numstages: Number of stages in CS tree
	%    maxheight: Number of levels in each stage
	%    wordlength: Input and output wordlength
	%    inbits: Input bitproducts, matrix of size (maxheight*numstages+1) x wordlength
	%    cterm: Constant term, vector of size 1 x wordlength
	%    constrained: Number of constrained bits in output
	%    costs: structure with fields:
	%      facost, hacost, regcost, bitcost
	
	%Copyright 2008, 2010 Anton Blad
	%
	%This file is part of firgen.
	%
	%firgen is free software: you can redistribute it and/or modify
	%it under the terms of the GNU General Public License as published by
	%the Free Software Foundation, either version 3 of the License, or
	%(at your option) any later version.
	%
	%firgen is distributed in the hope that it will be useful,
	%but WITHOUT ANY WARRANTY; without even the implied warranty of
	%MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	%GNU General Public License for more details.
	%
	%You should have received a copy of the GNU General Public License
	%along with firgen.  If not, see <http://www.gnu.org/licenses/>
	
	inbits = lpspec.inbits;
	cterm = lpspec.cterm;
	numstages = lpspec.numstages;
	maxheight = lpspec.maxheight;
	wordlength = lpspec.wordlength;
	constrained = lpspec.constrained;
	costs = lpspec.costs;
	
	% Generate BitsIn vector to optimization problem
	levelidx = reshape(repmat(0:maxheight*numstages-1, wordlength, 1), 1, maxheight*numstages*wordlength);
	bitidx = reshape(repmat(1:wordlength, maxheight*numstages, 1)', 1, maxheight*numstages*wordlength);
	bitproducts = reshape(inbits', 1, maxheight*numstages*wordlength);
	bitsin = reshape([levelidx;bitidx;bitproducts], 1, 3*maxheight*numstages*wordlength);
	
	ctermidx = 1:wordlength;
	ctermvec = reshape([ctermidx;cterm], 1, 2*wordlength);
	
	fid = fopen(datfile, 'w+');
	if fid == -1
		error 'gencslp: error opening file';
	end
	
	fprintf(fid, '/* Complexity costs */\n');
	fprintf(fid, 'param UCost_FA := %d;\n', costs.facost);
	fprintf(fid, 'param UCost_HA := %d;\n', costs.hacost);
	fprintf(fid, 'param UCost_reg := %d;\n', costs.regcost);
	fprintf(fid, 'param UCost_bit := %d;\n', costs.bitcost);
	fprintf(fid, '\n');
	fprintf(fid, 'param NumConstrainedOut := %d;\n', constrained);
	fprintf(fid, '\n');
	fprintf(fid, '/* Parameters */\n');
	fprintf(fid, 'param AdderLevels := %d;\n', maxheight*numstages);
	fprintf(fid, 'param Wordlength := %d;\n', wordlength);
	fprintf(fid, 'param MaxHeight := %d;\n', maxheight);
	fprintf(fid, 'param BitsIn := %s;\n', sprintf(' %d', bitsin));
	fprintf(fid, 'param CTerm := %s;\n', sprintf(' %d', ctermvec));
	fprintf(fid, '\n');
	fprintf(fid, 'end;\n');
	
	fclose(fid);
	
